using System;
using System.Globalization;
using System.Windows.Forms;

using gov.va.med.vbecs.Common;

namespace gov.va.med.vbecs.admin.GUI
{
	/// <summary>
	/// Provides basic UI validation for general purposes. 
	/// </summary>
	public class GuiValidator
	{
		private gov.va.med.vbecs.GUI.controls.VbecsErrorProvider _errorProvider; // **** 508 Compliance ***

		/// <summary>
		/// Constructor binding validator to form via supplied error provider.
		/// **** 508 Compliance ***
		/// </summary>
		/// <param name="boundErrorProvider">Bound error provider defined on the form validator is attached to.</param>
		public GuiValidator( gov.va.med.vbecs.GUI.controls.VbecsErrorProvider boundErrorProvider )
		{
			if( boundErrorProvider == null )
				throw( new ArgumentNullException( "boundErrorProvider" ) );

			_errorProvider = boundErrorProvider;
		}

		/// <summary>
		/// Verifies that the supplied text box contains valid value that 
		/// can be converted to <see cref="Int32"/>.
		/// </summary>
		/// <param name="textBox">Text box to verify.</param>
		/// <param name="requireValue">Specifies whether value in the text box is required or not.</param>
		/// <returns>True if text box contains valid string that can be converted to Int32.</returns>
		public bool VerifyInt32( TextBox textBox, bool requireValue )
		{
			if( textBox == null ) 
				throw( new ArgumentNullException( "textBox" ) );

			ClearValidationError( textBox );

			//CR 2936 - removing spaces from text
			if( !StUiNullConvert.IsZeroString( textBox.Text.Trim() ) )
			{
				double _result;

				if( Double.TryParse( textBox.Text, NumberStyles.Integer, NumberFormatInfo.InvariantInfo, out _result ) &&
					_result < StNullConvert.Int32MaxValue && 
					_result > StNullConvert.Int32MinValue )
					return true;
			
				SetValidationError( textBox, StrRes.ValidMsg.Common.InvalidIntegerEnteredPleaseProvideValid() );
				return false;
			}

			if( !requireValue )
				return true;

			SetValidationError( textBox, StrRes.ValidMsg.Common.NoIntegerEnteredPleaseProvide() );
			return false;
		}

		/// <summary>
		/// Verifies that the supplied numeric up-down control contains value that can be converted to <see cref="Int32"/>.
		/// </summary>
		/// <param name="numericUpd">Numeric up-down control to validate.</param>
		/// <returns>True if the numeric up-down contains value convertable to <see cref="Int32"/>. Otherwise - false.</returns>
		public bool VerifyInt32( NumericUpDown numericUpd )
		{
			if( numericUpd == null )
				throw( new ArgumentNullException( "numericUpd" ) );

			ClearValidationError( numericUpd );

			if( numericUpd.Value < StNullConvert.Int32MaxValue && numericUpd.Value > StNullConvert.Int32MinValue )
				return true;

			SetValidationError( numericUpd, StrRes.ValidMsg.Common.InvalidIntegerEnteredPleaseProvideValid() );

			return false;
		}

		/// <summary>
		/// Verifies that the string contained in supplied text box can be converted to <see cref="System.Net.IPAddress"/>. 
		/// May require or does not require a value to be entered depending on the requireValue parameter.
		/// </summary>
		/// <param name="textBox">Text box to validate.</param>
		/// <param name="requireValue">Specifies whether the text box has to contain string convertable to IP address or not.</param>
		/// <returns>True if the string in text box is convertable to <see cref="System.Net.IPAddress"/>. Otherwise - false.</returns>
		public bool VerifyIPAddress( TextBox textBox, bool requireValue )
		{
			if( textBox == null )
				throw( new ArgumentNullException( "textBox" ) );

			ClearValidationError( textBox );

			//CR 2936 - removing spaces from text
			if( StUiNullConvert.IsZeroString( textBox.Text.Trim() ) && !requireValue )
				return true;

			if( Common.RegularExpressions.IPAddress().IsMatch( textBox.Text.Trim() ) )
				return true;

			if (textBox.Text.Trim() == "")
			{
				if (textBox.Name.IndexOf("Interface") != -1)
					SetValidationError( textBox, Common.StrRes.Tooltips.MUC04.InterfaceIPAddressMustBeSpecified());
				else
					SetValidationError( textBox, Common.StrRes.Tooltips.MUC04.VBECSIPAddressMustBeSpecified());
			}
			else
				SetValidationError( textBox, StrRes.ValidMsg.Common.EnterValidIPAddress() );

			return false;
		}

		/// <summary>
		/// Verifies that the supplied text box contains a valid port number
		/// CR 2940
		/// </summary>
		/// <param name="textBox"></param>
		/// <param name="requireValue"></param>
		/// <returns></returns>
		public bool VerifyPortNumber( TextBox textBox, bool requireValue )
		{
			if( textBox == null ) 
				throw( new ArgumentNullException( "textBox" ) );

			ClearValidationError( textBox );

			if( !StUiNullConvert.IsZeroString( textBox.Text.Trim() ) )
			{
				double _result;

				if( Double.TryParse( textBox.Text, NumberStyles.Integer, NumberFormatInfo.InvariantInfo, out _result ) &&
					_result < StNullConvert.Int32MaxValue && 
					_result > StNullConvert.Int32MinValue )
				{
					if (Convert.ToInt32(textBox.Text.Trim()) >= 1024 && Convert.ToInt32(textBox.Text.Trim()) <= 65535)
						return true;
				}
			
				SetValidationError( textBox, StrRes.ValidMsg.MUC01.InvalidPortNumber(1024, 65535));
				return false;
			}
			else
			{
				if(!requireValue )
				{
					return true;
				}
					//CR 2952, CR 2954 Added tooltip for situations when Port Number is blank
				else
				{
					SetValidationError( textBox, StrRes.Tooltips.MUC04.PortNumberMustBeSpecified() );
					return false;
				}
			}
		}

		/// <summary>
		/// Verifies that the string contained in supplied text box can be converted to <see cref="System.Net.IPAddress"/>. 
		/// Does not require a value (text box may be empty). 
		/// </summary>
		/// <param name="textBox">Text box to validate.</param>
		/// <returns>True if the string in text box is convertable to <see cref="System.Net.IPAddress"/> or empty. Otherwise - false.</returns>
		public bool VerifyIPAddress( TextBox textBox )
		{
			return VerifyIPAddress( textBox, false );
		}

		/// <summary>
		/// Verifies that the supplied numeric up-down control contains value that can be converted to <see cref="Byte"/>.
		/// </summary>
		/// <param name="numericUpd">Numeric up-down control to validate.</param>
		/// <returns>True if the numeric up-down contains value convertable to <see cref="Byte"/>. Otherwise - false.</returns>
		public bool VerifyByte( NumericUpDown numericUpd )
		{
			if( numericUpd == null )
				throw( new ArgumentNullException( "numericUpd" ) );

			ClearValidationError( numericUpd );

			if( numericUpd.Value < StNullConvert.ByteMaxValue && numericUpd.Value > StNullConvert.ByteMinValue )
				return true;

			SetValidationError( numericUpd, StrRes.ValidMsg.Common.InvalidByteEnteredPleaseProvideValid( Byte.MinValue, Byte.MaxValue ) );

			return false;
		}

		/// <summary>
		/// Sets validation error message for a given control via bound error provider. 
		/// </summary>
		/// <param name="control">Control to set validation error message for.</param>
		/// <param name="errorMessage">Validation error message.</param>
		public virtual void SetValidationError( Control control, string errorMessage )
		{
			if( control == null )
				throw( new ArgumentNullException( "control" ) );

			if (control.Enabled)
			{
				_errorProvider.SetVbecsError( control, errorMessage ); // **** 508 Compliance ***
			}
		}

		/// <summary>
		/// Sets validation error message for a given control via bound error provider. 
		/// </summary>
		/// <param name="control">Control to set validation error message for.</param>
		/// <param name="errorMessageItem">Validation error message string resource item.</param>
		public virtual void SetValidationError( Control control, StrRes.StringResourceItem errorMessageItem )
		{
			SetValidationError( control, errorMessageItem.ResString );
		}

		/// <summary>
		/// Clears validation error for supplied control from bound error provider.
		/// </summary>
		/// <param name="control">Control to clear validation error for.</param>
		public virtual void ClearValidationError( Control control )
		{
			if( control == null )
				throw( new ArgumentNullException( "control" ) );

			_errorProvider.SetVbecsError( control, null ); // **** 508 Compliance ***
		}

		/// <summary>
		/// Returns error message from control
		/// </summary>
		/// <param name="control"></param>
		/// <returns></returns>
		public string GetValidationError(Control control)
		{
			return _errorProvider.GetError(control);
		}
	}
}
